const bcrypt = require('bcrypt');
const { sequelize } = require('../config/database');
const { SuperAdmin, Hotel, Category } = require('../models');
require('dotenv').config();

/**
 * Initialize database with super admin and demo hotel
 */
const initializeDatabase = async () => {
    try {
        console.log('🔄 Initializing database...\n');

        // Connect to database
        await sequelize.authenticate();
        console.log('✅ Database connection established');

        // Create tables
        await sequelize.sync({ force: false, alter: true });
        console.log('✅ Database tables created/updated\n');

        // Create Super Admin if not exists
        const superAdminEmail = process.env.SUPER_ADMIN_EMAIL || 'superadmin@hotel.com';
        const superAdminPassword = process.env.SUPER_ADMIN_PASSWORD || 'SuperAdmin@123';

        const existingSuperAdmin = await SuperAdmin.findOne({
            where: { email: superAdminEmail }
        });

        if (!existingSuperAdmin) {
            const passwordHash = await bcrypt.hash(superAdminPassword, 10);
            await SuperAdmin.create({
                email: superAdminEmail,
                passwordHash
            });
            console.log('✅ Super Admin created');
            console.log(`   Email: ${superAdminEmail}`);
            console.log(`   Password: ${superAdminPassword}\n`);
        } else {
            console.log('ℹ️  Super Admin already exists\n');
        }

        // Create Demo Hotel if not exists
        const demoHotelEmail = 'demo@hotel.com';
        const existingHotel = await Hotel.findOne({
            where: { email: demoHotelEmail }
        });

        if (!existingHotel) {
            const passwordHash = await bcrypt.hash('demo123', 10);

            // Create subscription end date (1 year from now)
            const subscriptionEnd = new Date();
            subscriptionEnd.setFullYear(subscriptionEnd.getFullYear() + 1);

            const demoHotel = await Hotel.create({
                name: 'Grand Horizon Hotel',
                email: demoHotelEmail,
                passwordHash,
                subscriptionEnd,
                isActive: true,
                tagline: 'Experience Fine Dining',
                description: 'Welcome to Grand Horizon Hotel. We serve the finest cuisine with exceptional service.',
                address: '123 Main Street, City Center',
                phone: '+1234567890',
                whatsappNumber: '+1234567890',
                logoUrl: 'https://via.placeholder.com/100',
                bannerUrl: 'https://via.placeholder.com/1200x400'
            });

            console.log('✅ Demo Hotel created');
            console.log(`   Name: ${demoHotel.name}`);
            console.log(`   Email: ${demoHotelEmail}`);
            console.log(`   Password: demo123`);
            console.log(`   Hotel ID: ${demoHotel.id}\n`);

            // Create default categories for demo hotel
            const defaultCategories = [
                { name: 'Starters', mealTimes: ['Lunch', 'Dinner'] },
                { name: 'Main Course', mealTimes: ['Lunch', 'Dinner'] },
                { name: 'Beverages', mealTimes: ['Breakfast', 'Lunch', 'Dinner'] },
                { name: 'Desserts', mealTimes: ['Lunch', 'Dinner'] }
            ];

            await Promise.all(
                defaultCategories.map(cat =>
                    Category.create({
                        hotelId: demoHotel.id,
                        name: cat.name,
                        mealTimes: cat.mealTimes
                    })
                )
            );

            console.log('✅ Default categories created for demo hotel\n');
        } else {
            console.log('ℹ️  Demo Hotel already exists\n');
        }

        console.log('🎉 Database initialization complete!\n');
        console.log('📝 You can now:');
        console.log('   1. Start the server: npm start');
        console.log('   2. Login as Super Admin to create hotels');
        console.log('   3. Login as Demo Hotel to manage menu\n');

        process.exit(0);
    } catch (error) {
        console.error('❌ Database initialization failed:', error);
        process.exit(1);
    }
};

initializeDatabase();
