const path = require('path');
const { Hotel } = require(path.join(__dirname, '..', 'models'));
const { sequelize } = require(path.join(__dirname, '..', 'config', 'database'));
require('dotenv').config({ path: path.join(__dirname, '..', '.env') });

async function checkSubscription() {
    try {
        await sequelize.authenticate();
        console.log('✅ Connected to database\n');

        // Find demo hotel
        const demoHotel = await Hotel.findOne({
            where: { email: 'demo@hotel.com' },
            raw: true
        });

        if (!demoHotel) {
            console.log('❌ Demo hotel not found!');
            process.exit(1);
        }

        const now = new Date();
        const subEnd = new Date(demoHotel.subscriptionEnd);
        const diffTime = subEnd.getTime() - now.getTime();
        const daysRemaining = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        const isExpired = subEnd < now;

        console.log('📊 Demo Hotel Subscription Status:\n');
        console.log('Hotel Details:');
        console.log(`   Name: ${demoHotel.name}`);
        console.log(`   Email: ${demoHotel.email}`);
        console.log(`   ID: ${demoHotel.id}`);
        console.log(`   Active: ${demoHotel.isActive ? 'Yes' : 'No'}\n`);

        console.log('Subscription:');
        console.log(`   End Date: ${subEnd.toLocaleString()}`);
        console.log(`   Current Time: ${now.toLocaleString()}`);
        console.log(`   Days Remaining: ${daysRemaining}`);
        console.log(`   Status: ${isExpired ? '❌ EXPIRED' : '✅ ACTIVE'}\n`);

        if (isExpired) {
            console.log('⚠️  SUBSCRIPTION IS EXPIRED!');
            console.log('   Run: npm run extend-demo\n');
        } else {
            console.log('✅ Subscription is active!');
            console.log('   You should be able to login.\n');
        }

        // Show raw database value
        console.log('Raw Database Value:');
        console.log(`   subscriptionEnd: ${demoHotel.subscriptionEnd}\n`);

        process.exit(0);
    } catch (error) {
        console.error('❌ Error:', error);
        process.exit(1);
    }
}

checkSubscription();
