const express = require('express');
const router = express.Router();
const { getHotelInfo } = require('../controllers/publicController');
const { getPublicMenu, getPublicCategories } = require('../controllers/menuController');
const { Offer } = require('../models');

// Get hotel information
router.get('/:hotelId/info', getHotelInfo);

// Get menu items
router.get('/:hotelId/menu', getPublicMenu);

// Get categories
router.get('/:hotelId/categories', getPublicCategories);

// Get active offers
router.get('/:hotelId/offers', async (req, res) => {
    try {
        const { hotelId } = req.params;
        const { Hotel } = require('../models');

        // Check hotel status
        const hotel = await Hotel.findByPk(hotelId);
        if (!hotel) {
            return res.status(404).json({ success: false, message: 'Hotel not found' });
        }

        const now = new Date();
        const subscriptionEnd = new Date(hotel.subscriptionEnd);
        if (!hotel.isActive || subscriptionEnd < now) {
            return res.status(403).json({ success: false, message: 'Hotel service suspended' });
        }

        const offers = await Offer.findAll({
            where: {
                hotelId,
                startDate: { [require('sequelize').Op.lte]: now },
                endDate: { [require('sequelize').Op.gte]: now }
            },
            order: [['createdAt', 'DESC']]
        });

        const formattedOffers = offers.map(offer => ({
            id: offer.id,
            name: offer.name,
            description: offer.description,
            discountType: offer.discountType,
            discountValue: parseFloat(offer.discountValue),
            startDate: offer.startDate,
            endDate: offer.endDate,
            targetMealTime: offer.targetMealTime,
            targetCategory: offer.targetCategory
        }));

        res.json({
            success: true,
            data: formattedOffers
        });
    } catch (error) {
        console.error('Get offers error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
});

module.exports = router;
