const express = require('express');
const router = express.Router();
const { authenticateHotelAdmin } = require('../middleware/auth');
const { upload, handleUploadError } = require('../middleware/upload');
const {
    getMenuItems,
    createMenuItem,
    updateMenuItem,
    deleteMenuItem,
    getCategories,
    createCategory,
    updateCategory,
    deleteCategory
} = require('../controllers/adminController');
const { updateHotelSettings, getDashboardStats } = require('../controllers/hotelController');
const { Offer } = require('../models');

// All routes require hotel admin authentication
router.use(authenticateHotelAdmin);

// Dashboard
router.get('/dashboard', getDashboardStats);

// Menu Items
router.get('/menu', getMenuItems);
router.post('/menu', upload.single('image'), handleUploadError, createMenuItem);
router.put('/menu/:id', upload.single('image'), handleUploadError, updateMenuItem);
router.delete('/menu/:id', deleteMenuItem);

// Categories
router.get('/categories', getCategories);
router.post('/categories', createCategory);
router.put('/categories/:id', updateCategory);
router.delete('/categories/:id', deleteCategory);

// Offers
router.get('/offers', async (req, res) => {
    try {
        const hotelId = req.user.hotelId;

        const offers = await Offer.findAll({
            where: { hotelId },
            order: [['createdAt', 'DESC']]
        });

        const formattedOffers = offers.map(offer => ({
            id: offer.id,
            name: offer.name,
            description: offer.description,
            discountType: offer.discountType,
            discountValue: parseFloat(offer.discountValue),
            startDate: offer.startDate,
            endDate: offer.endDate,
            targetMealTime: offer.targetMealTime,
            targetCategory: offer.targetCategory
        }));

        res.json({
            success: true,
            data: formattedOffers
        });
    } catch (error) {
        console.error('Get offers error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
});

router.post('/offers', async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const {
            name, description, discountType, discountValue,
            startDate, endDate, targetMealTime, targetCategory
        } = req.body;

        if (!name || !discountType || !discountValue || !startDate || !endDate) {
            return res.status(400).json({
                success: false,
                message: 'Required fields missing'
            });
        }

        const offer = await Offer.create({
            hotelId,
            name,
            description,
            discountType,
            discountValue,
            startDate,
            endDate,
            targetMealTime: targetMealTime || null,
            targetCategory: targetCategory || null
        });

        res.status(201).json({
            success: true,
            message: 'Offer created successfully',
            data: {
                id: offer.id,
                name: offer.name,
                description: offer.description,
                discountType: offer.discountType,
                discountValue: parseFloat(offer.discountValue),
                startDate: offer.startDate,
                endDate: offer.endDate,
                targetMealTime: offer.targetMealTime,
                targetCategory: offer.targetCategory
            }
        });
    } catch (error) {
        console.error('Create offer error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
});

router.put('/offers/:id', async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const { id } = req.params;
        const {
            name, description, discountType, discountValue,
            startDate, endDate, targetMealTime, targetCategory
        } = req.body;

        const offer = await Offer.findOne({
            where: { id, hotelId }
        });

        if (!offer) {
            return res.status(404).json({
                success: false,
                message: 'Offer not found'
            });
        }

        await offer.update({
            name: name || offer.name,
            description: description !== undefined ? description : offer.description,
            discountType: discountType || offer.discountType,
            discountValue: discountValue || offer.discountValue,
            startDate: startDate || offer.startDate,
            endDate: endDate || offer.endDate,
            targetMealTime: targetMealTime !== undefined ? targetMealTime : offer.targetMealTime,
            targetCategory: targetCategory !== undefined ? targetCategory : offer.targetCategory
        });

        res.json({
            success: true,
            message: 'Offer updated successfully',
            data: {
                id: offer.id,
                name: offer.name,
                description: offer.description,
                discountType: offer.discountType,
                discountValue: parseFloat(offer.discountValue),
                startDate: offer.startDate,
                endDate: offer.endDate,
                targetMealTime: offer.targetMealTime,
                targetCategory: offer.targetCategory
            }
        });
    } catch (error) {
        console.error('Update offer error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
});

router.delete('/offers/:id', async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const { id } = req.params;

        const offer = await Offer.findOne({
            where: { id, hotelId }
        });

        if (!offer) {
            return res.status(404).json({
                success: false,
                message: 'Offer not found'
            });
        }

        await offer.destroy();

        res.json({
            success: true,
            message: 'Offer deleted successfully'
        });
    } catch (error) {
        console.error('Delete offer error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
});

// Hotel Settings
router.put('/settings',
    upload.fields([
        { name: 'logo', maxCount: 1 },
        { name: 'banner', maxCount: 1 }
    ]),
    handleUploadError,
    updateHotelSettings
);

module.exports = router;
