const bcrypt = require('bcrypt');
const { Hotel, MenuItem, Category, Offer } = require('../models');

/**
 * Get all hotels
 */
const getAllHotels = async (req, res) => {
    try {
        const hotels = await Hotel.findAll({
            attributes: [
                'id', 'name', 'email', 'subscriptionEnd',
                'isActive', 'createdAt', 'phone', 'address'
            ],
            order: [['createdAt', 'DESC']]
        });

        const hotelsWithStats = await Promise.all(
            hotels.map(async (hotel) => {
                const [itemCount, categoryCount] = await Promise.all([
                    MenuItem.count({ where: { hotelId: hotel.id } }),
                    Category.count({ where: { hotelId: hotel.id } })
                ]);

                const now = new Date();
                const subscriptionEnd = new Date(hotel.subscriptionEnd);
                const diffTime = subscriptionEnd.getTime() - now.getTime();
                const remainingDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

                return {
                    id: hotel.id,
                    name: hotel.name,
                    email: hotel.email,
                    subscriptionEnd: hotel.subscriptionEnd,
                    remainingDays,
                    isActive: hotel.isActive,
                    createdAt: hotel.createdAt,
                    phone: hotel.phone,
                    address: hotel.address,
                    stats: {
                        menuItems: itemCount,
                        categories: categoryCount
                    }
                };
            })
        );

        res.json({
            success: true,
            data: hotelsWithStats
        });
    } catch (error) {
        console.error('Get all hotels error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Create a new hotel account
 */
const createHotel = async (req, res) => {
    try {
        const { name, email, password, subscriptionDays } = req.body;

        if (!name || !email || !password) {
            return res.status(400).json({
                success: false,
                message: 'Name, email, and password are required'
            });
        }

        // Check if email already exists
        const existingHotel = await Hotel.findOne({ where: { email } });
        if (existingHotel) {
            return res.status(400).json({
                success: false,
                message: 'Email already exists'
            });
        }

        // Hash password
        const passwordHash = await bcrypt.hash(password, 10);

        // Calculate subscription end date
        const days = subscriptionDays || 30; // Default 30 days
        const subscriptionEnd = new Date();
        subscriptionEnd.setDate(subscriptionEnd.getDate() + days);

        // Create hotel
        const hotel = await Hotel.create({
            name,
            email,
            passwordHash,
            subscriptionEnd,
            isActive: true,
            tagline: 'Welcome to ' + name,
            description: 'Best food in town'
        });

        // Create default categories
        const defaultCategories = [
            { name: 'Starters', mealTimes: ['Lunch', 'Dinner'] },
            { name: 'Main Course', mealTimes: ['Lunch', 'Dinner'] },
            { name: 'Beverages', mealTimes: ['Breakfast', 'Lunch', 'Dinner'] },
            { name: 'Desserts', mealTimes: ['Lunch', 'Dinner'] }
        ];

        await Promise.all(
            defaultCategories.map(cat =>
                Category.create({
                    hotelId: hotel.id,
                    name: cat.name,
                    mealTimes: cat.mealTimes
                })
            )
        );

        res.status(201).json({
            success: true,
            message: 'Hotel created successfully',
            data: {
                id: hotel.id,
                name: hotel.name,
                email: hotel.email,
                subscriptionEnd: hotel.subscriptionEnd,
                isActive: hotel.isActive
            }
        });
    } catch (error) {
        console.error('Create hotel error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Update hotel subscription
 */
const updateHotelSubscription = async (req, res) => {
    try {
        const { id } = req.params;
        const { subscriptionDays, endDate } = req.body;

        if ((!subscriptionDays || subscriptionDays < 1) && !endDate) {
            return res.status(400).json({
                success: false,
                message: 'Valid subscription days or end date required'
            });
        }

        const hotel = await Hotel.findByPk(id);

        if (!hotel) {
            return res.status(404).json({
                success: false,
                message: 'Hotel not found'
            });
        }

        let newSubscriptionEnd;

        if (endDate) {
            // Set exact date
            newSubscriptionEnd = new Date(endDate);
            // Ensure end of day
            newSubscriptionEnd.setHours(23, 59, 59, 999);

            if (newSubscriptionEnd < new Date()) {
                return res.status(400).json({
                    success: false,
                    message: 'End date cannot be in the past'
                });
            }
        } else {
            // Extend from current end date or now, whichever is later
            const now = new Date();
            const currentEnd = new Date(hotel.subscriptionEnd);
            const baseDate = currentEnd > now ? currentEnd : now;

            newSubscriptionEnd = new Date(baseDate);
            newSubscriptionEnd.setDate(newSubscriptionEnd.getDate() + subscriptionDays);
        }

        await hotel.update({
            subscriptionEnd: newSubscriptionEnd
        });

        res.json({
            success: true,
            message: `Subscription extended by ${subscriptionDays} days`,
            data: {
                id: hotel.id,
                name: hotel.name,
                subscriptionEnd: hotel.subscriptionEnd
            }
        });
    } catch (error) {
        console.error('Update subscription error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Toggle hotel active status
 */
const toggleHotelStatus = async (req, res) => {
    try {
        const { id } = req.params;

        const hotel = await Hotel.findByPk(id);

        if (!hotel) {
            return res.status(404).json({
                success: false,
                message: 'Hotel not found'
            });
        }

        await hotel.update({
            isActive: !hotel.isActive
        });

        res.json({
            success: true,
            message: `Hotel ${hotel.isActive ? 'activated' : 'suspended'}`,
            data: {
                id: hotel.id,
                name: hotel.name,
                isActive: hotel.isActive
            }
        });
    } catch (error) {
        console.error('Toggle hotel status error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Delete a hotel (and all its data)
 */
const deleteHotel = async (req, res) => {
    try {
        const { id } = req.params;

        const hotel = await Hotel.findByPk(id);

        if (!hotel) {
            return res.status(404).json({
                success: false,
                message: 'Hotel not found'
            });
        }

        // Sequelize will cascade delete all related data
        await hotel.destroy();

        res.json({
            success: true,
            message: 'Hotel and all associated data deleted successfully'
        });
    } catch (error) {
        console.error('Delete hotel error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

module.exports = {
    getAllHotels,
    createHotel,
    updateHotelSubscription,
    toggleHotelStatus,
    deleteHotel
};
