const { MenuItem, Category } = require('../models');
const { Op } = require('sequelize');

/**
 * Get all menu items for a hotel (public)
 */
const getPublicMenu = async (req, res) => {
    try {
        const { hotelId } = req.params;
        const { Hotel } = require('../models');

        // Check hotel status
        const hotel = await Hotel.findByPk(hotelId);
        if (!hotel) {
            return res.status(404).json({ success: false, message: 'Hotel not found' });
        }

        const now = new Date();
        const subscriptionEnd = new Date(hotel.subscriptionEnd);
        if (!hotel.isActive || subscriptionEnd < now) {
            return res.status(403).json({ success: false, message: 'Hotel service suspended' });
        }

        const menuItems = await MenuItem.findAll({
            where: {
                hotelId,
                isVisible: true
            },
            include: [{
                model: Category,
                as: 'category',
                attributes: ['id', 'name', 'mealTimes']
            }],
            order: [['createdAt', 'DESC']]
        });

        // Transform to match frontend format
        const formattedItems = menuItems.map(item => ({
            id: item.id,
            name: item.name,
            description: item.description,
            price: parseFloat(item.price),
            offerPrice: item.offerPrice ? parseFloat(item.offerPrice) : undefined,
            image: item.imageUrl || 'https://via.placeholder.com/300x200',
            mealTime: item.mealTime,
            category: item.category.name,
            dietary: item.dietaryType,
            isSpecial: item.isSpecial,
            isVisible: item.isVisible,
            availableFrom: item.availableFrom,
            availableTo: item.availableTo
        }));

        res.json({
            success: true,
            data: formattedItems
        });
    } catch (error) {
        console.error('Get public menu error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Get all categories for a hotel (public)
 */
const getPublicCategories = async (req, res) => {
    try {
        const { hotelId } = req.params;
        const { Hotel } = require('../models');

        // Check hotel status
        const hotel = await Hotel.findByPk(hotelId);
        if (!hotel) {
            return res.status(404).json({ success: false, message: 'Hotel not found' });
        }

        const now = new Date();
        const subscriptionEnd = new Date(hotel.subscriptionEnd);
        if (!hotel.isActive || subscriptionEnd < now) {
            return res.status(403).json({ success: false, message: 'Hotel service suspended' });
        }

        const categories = await Category.findAll({
            where: { hotelId },
            attributes: ['id', 'name', 'mealTimes'],
            order: [['createdAt', 'ASC']]
        });

        // Transform to match frontend format
        const formattedCategories = categories.map(cat => ({
            id: cat.id.toString(),
            name: cat.name,
            mealTime: cat.mealTimes
        }));

        res.json({
            success: true,
            data: formattedCategories
        });
    } catch (error) {
        console.error('Get public categories error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

module.exports = {
    getPublicMenu,
    getPublicCategories
};
