const { Hotel } = require('../models');
const path = require('path');
const fs = require('fs');

/**
 * Update hotel settings
 */
const updateHotelSettings = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const {
            name, tagline, description, address, phone, whatsappNumber
        } = req.body;

        const hotel = await Hotel.findByPk(hotelId);

        if (!hotel) {
            return res.status(404).json({
                success: false,
                message: 'Hotel not found'
            });
        }

        // Handle logo upload
        let logoUrl = hotel.logoUrl;
        if (req.files && req.files.logo) {
            const logoFile = req.files.logo[0];
            // Delete old logo if it exists
            if (hotel.logoUrl && !hotel.logoUrl.includes('placeholder')) {
                const oldLogoPath = path.join(__dirname, '..', 'public', 'uploads', path.basename(hotel.logoUrl));
                if (fs.existsSync(oldLogoPath)) {
                    fs.unlinkSync(oldLogoPath);
                }
            }
            logoUrl = `${req.protocol}://${req.get('host')}/uploads/${logoFile.filename}`;
        }

        // Handle banner upload
        let bannerUrl = hotel.bannerUrl;
        if (req.files && req.files.banner) {
            const bannerFile = req.files.banner[0];
            // Delete old banner if it exists
            if (hotel.bannerUrl && !hotel.bannerUrl.includes('placeholder')) {
                const oldBannerPath = path.join(__dirname, '..', 'public', 'uploads', path.basename(hotel.bannerUrl));
                if (fs.existsSync(oldBannerPath)) {
                    fs.unlinkSync(oldBannerPath);
                }
            }
            bannerUrl = `${req.protocol}://${req.get('host')}/uploads/${bannerFile.filename}`;
        }

        // Update hotel
        await hotel.update({
            name: name || hotel.name,
            tagline: tagline !== undefined ? tagline : hotel.tagline,
            description: description !== undefined ? description : hotel.description,
            address: address !== undefined ? address : hotel.address,
            phone: phone !== undefined ? phone : hotel.phone,
            whatsappNumber: whatsappNumber !== undefined ? whatsappNumber : hotel.whatsappNumber,
            logoUrl,
            bannerUrl
        });

        res.json({
            success: true,
            message: 'Hotel settings updated successfully',
            data: {
                name: hotel.name,
                tagline: hotel.tagline,
                description: hotel.description,
                address: hotel.address,
                phone: hotel.phone,
                whatsapp: hotel.whatsappNumber,
                logo: hotel.logoUrl,
                banner: hotel.bannerUrl
            }
        });
    } catch (error) {
        console.error('Update hotel settings error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Get hotel dashboard stats
 */
const getDashboardStats = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const { MenuItem, Category, Offer } = require('../models');

        const [totalItems, totalCategories, totalOffers, visibleItems] = await Promise.all([
            MenuItem.count({ where: { hotelId } }),
            Category.count({ where: { hotelId } }),
            Offer.count({ where: { hotelId } }),
            MenuItem.count({ where: { hotelId, isVisible: true } })
        ]);

        // Get subscription info
        const hotel = await Hotel.findByPk(hotelId);
        const now = new Date();
        const subscriptionEnd = new Date(hotel.subscriptionEnd);
        const diffTime = subscriptionEnd.getTime() - now.getTime();
        const remainingDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

        res.json({
            success: true,
            data: {
                totalItems,
                totalCategories,
                totalOffers,
                visibleItems,
                subscription: {
                    endDate: hotel.subscriptionEnd,
                    remainingDays,
                    isActive: hotel.isActive && remainingDays > 0
                },
                hotel: {
                    name: hotel.name,
                    tagline: hotel.tagline,
                    description: hotel.description,
                    address: hotel.address,
                    phone: hotel.phone,
                    whatsappNumber: hotel.whatsappNumber,
                    logoUrl: hotel.logoUrl,
                    bannerUrl: hotel.bannerUrl
                }
            }
        });
    } catch (error) {
        console.error('Get dashboard stats error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

module.exports = {
    updateHotelSettings,
    getDashboardStats
};
