const { MenuItem, Category } = require('../models');
const path = require('path');
const fs = require('fs');

/**
 * Get all menu items for the logged-in hotel
 */
const getMenuItems = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;

        const menuItems = await MenuItem.findAll({
            where: { hotelId },
            include: [{
                model: Category,
                as: 'category',
                attributes: ['id', 'name', 'mealTimes']
            }],
            order: [['createdAt', 'DESC']]
        });

        const formattedItems = menuItems.map(item => ({
            id: item.id,
            name: item.name,
            description: item.description,
            price: parseFloat(item.price),
            offerPrice: item.offerPrice ? parseFloat(item.offerPrice) : undefined,
            image: item.imageUrl || 'https://via.placeholder.com/300x200',
            mealTime: item.mealTime,
            category: item.category.name,
            categoryId: item.categoryId,
            dietary: item.dietaryType,
            isSpecial: item.isSpecial,
            isVisible: item.isVisible,
            availableFrom: item.availableFrom,
            availableTo: item.availableTo
        }));

        res.json({
            success: true,
            data: formattedItems
        });
    } catch (error) {
        console.error('Get menu items error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Create a new menu item
 */
const createMenuItem = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const {
            name, description, price, offerPrice, mealTime,
            categoryId, dietary, isSpecial, isVisible,
            availableFrom, availableTo
        } = req.body;

        // Verify category belongs to this hotel
        const category = await Category.findOne({
            where: { id: categoryId, hotelId }
        });

        if (!category) {
            return res.status(400).json({
                success: false,
                message: 'Invalid category'
            });
        }

        // Handle image upload
        let imageUrl = null;
        if (req.file) {
            imageUrl = `${req.protocol}://${req.get('host')}/uploads/${req.file.filename}`;
        }

        const menuItem = await MenuItem.create({
            hotelId,
            categoryId,
            name,
            description,
            price,
            offerPrice: offerPrice || null,
            imageUrl,
            mealTime,
            dietaryType: dietary || 'None',
            isSpecial: isSpecial === 'true' || isSpecial === true,
            isVisible: isVisible === 'true' || isVisible === true,
            availableFrom: availableFrom || null,
            availableTo: availableTo || null
        });

        res.status(201).json({
            success: true,
            message: 'Menu item created successfully',
            data: {
                id: menuItem.id,
                name: menuItem.name,
                description: menuItem.description,
                price: parseFloat(menuItem.price),
                offerPrice: menuItem.offerPrice ? parseFloat(menuItem.offerPrice) : undefined,
                image: menuItem.imageUrl,
                mealTime: menuItem.mealTime,
                category: category.name,
                categoryId: menuItem.categoryId,
                dietary: menuItem.dietaryType,
                isSpecial: menuItem.isSpecial,
                isVisible: menuItem.isVisible,
                availableFrom: menuItem.availableFrom,
                availableTo: menuItem.availableTo
            }
        });
    } catch (error) {
        console.error('Create menu item error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Update a menu item
 */
const updateMenuItem = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const { id } = req.params;
        const {
            name, description, price, offerPrice, mealTime,
            categoryId, dietary, isSpecial, isVisible,
            availableFrom, availableTo
        } = req.body;

        // Find item and verify ownership
        const menuItem = await MenuItem.findOne({
            where: { id, hotelId }
        });

        if (!menuItem) {
            return res.status(404).json({
                success: false,
                message: 'Menu item not found'
            });
        }

        // If category is being changed, verify it belongs to this hotel
        if (categoryId && categoryId !== menuItem.categoryId) {
            const category = await Category.findOne({
                where: { id: categoryId, hotelId }
            });

            if (!category) {
                return res.status(400).json({
                    success: false,
                    message: 'Invalid category'
                });
            }
        }

        // Handle image upload
        let imageUrl = menuItem.imageUrl;
        if (req.file) {
            // Delete old image if it exists and is not a placeholder
            if (menuItem.imageUrl && !menuItem.imageUrl.includes('placeholder')) {
                const oldImagePath = path.join(__dirname, '..', 'public', 'uploads', path.basename(menuItem.imageUrl));
                if (fs.existsSync(oldImagePath)) {
                    fs.unlinkSync(oldImagePath);
                }
            }
            imageUrl = `${req.protocol}://${req.get('host')}/uploads/${req.file.filename}`;
        }

        // Update item
        await menuItem.update({
            name: name || menuItem.name,
            description: description !== undefined ? description : menuItem.description,
            price: price || menuItem.price,
            offerPrice: offerPrice !== undefined ? offerPrice : menuItem.offerPrice,
            imageUrl,
            mealTime: mealTime || menuItem.mealTime,
            categoryId: categoryId || menuItem.categoryId,
            dietaryType: dietary || menuItem.dietaryType,
            isSpecial: isSpecial !== undefined ? (isSpecial === 'true' || isSpecial === true) : menuItem.isSpecial,
            isVisible: isVisible !== undefined ? (isVisible === 'true' || isVisible === true) : menuItem.isVisible,
            availableFrom: availableFrom !== undefined ? availableFrom : menuItem.availableFrom,
            availableTo: availableTo !== undefined ? availableTo : menuItem.availableTo
        });

        // Fetch updated item with category
        const updatedItem = await MenuItem.findByPk(id, {
            include: [{
                model: Category,
                as: 'category',
                attributes: ['id', 'name']
            }]
        });

        res.json({
            success: true,
            message: 'Menu item updated successfully',
            data: {
                id: updatedItem.id,
                name: updatedItem.name,
                description: updatedItem.description,
                price: parseFloat(updatedItem.price),
                offerPrice: updatedItem.offerPrice ? parseFloat(updatedItem.offerPrice) : undefined,
                image: updatedItem.imageUrl,
                mealTime: updatedItem.mealTime,
                category: updatedItem.category.name,
                categoryId: updatedItem.categoryId,
                dietary: updatedItem.dietaryType,
                isSpecial: updatedItem.isSpecial,
                isVisible: updatedItem.isVisible,
                availableFrom: updatedItem.availableFrom,
                availableTo: updatedItem.availableTo
            }
        });
    } catch (error) {
        console.error('Update menu item error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Delete a menu item
 */
const deleteMenuItem = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const { id } = req.params;

        const menuItem = await MenuItem.findOne({
            where: { id, hotelId }
        });

        if (!menuItem) {
            return res.status(404).json({
                success: false,
                message: 'Menu item not found'
            });
        }

        // Delete image file if it exists
        if (menuItem.imageUrl && !menuItem.imageUrl.includes('placeholder')) {
            const imagePath = path.join(__dirname, '..', 'public', 'uploads', path.basename(menuItem.imageUrl));
            if (fs.existsSync(imagePath)) {
                fs.unlinkSync(imagePath);
            }
        }

        await menuItem.destroy();

        res.json({
            success: true,
            message: 'Menu item deleted successfully'
        });
    } catch (error) {
        console.error('Delete menu item error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Get all categories for the logged-in hotel
 */
const getCategories = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;

        const categories = await Category.findAll({
            where: { hotelId },
            order: [['createdAt', 'ASC']]
        });

        const formattedCategories = categories.map(cat => ({
            id: cat.id.toString(),
            name: cat.name,
            mealTime: cat.mealTimes
        }));

        res.json({
            success: true,
            data: formattedCategories
        });
    } catch (error) {
        console.error('Get categories error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Create a new category
 */
const createCategory = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const { name, mealTime } = req.body;

        if (!name || !mealTime || !Array.isArray(mealTime)) {
            return res.status(400).json({
                success: false,
                message: 'Name and mealTime array are required'
            });
        }

        const category = await Category.create({
            hotelId,
            name,
            mealTimes: mealTime
        });

        res.status(201).json({
            success: true,
            message: 'Category created successfully',
            data: {
                id: category.id.toString(),
                name: category.name,
                mealTime: category.mealTimes
            }
        });
    } catch (error) {
        console.error('Create category error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Update a category
 */
const updateCategory = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const { id } = req.params;
        const { name, mealTime } = req.body;

        const category = await Category.findOne({
            where: { id, hotelId }
        });

        if (!category) {
            return res.status(404).json({
                success: false,
                message: 'Category not found'
            });
        }

        await category.update({
            name: name || category.name,
            mealTimes: mealTime || category.mealTimes
        });

        res.json({
            success: true,
            message: 'Category updated successfully',
            data: {
                id: category.id.toString(),
                name: category.name,
                mealTime: category.mealTimes
            }
        });
    } catch (error) {
        console.error('Update category error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

/**
 * Delete a category
 */
const deleteCategory = async (req, res) => {
    try {
        const hotelId = req.user.hotelId;
        const { id } = req.params;

        const category = await Category.findOne({
            where: { id, hotelId }
        });

        if (!category) {
            return res.status(404).json({
                success: false,
                message: 'Category not found'
            });
        }

        // Check if category has items
        const itemCount = await MenuItem.count({
            where: { categoryId: id }
        });

        if (itemCount > 0) {
            return res.status(400).json({
                success: false,
                message: `Cannot delete category with ${itemCount} items. Please reassign or delete items first.`
            });
        }

        await category.destroy();

        res.json({
            success: true,
            message: 'Category deleted successfully'
        });
    } catch (error) {
        console.error('Delete category error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error'
        });
    }
};

module.exports = {
    getMenuItems,
    createMenuItem,
    updateMenuItem,
    deleteMenuItem,
    getCategories,
    createCategory,
    updateCategory,
    deleteCategory
};
