# Hotel Ordering System - Backend

A production-ready Node.js + Express + MySQL backend for a multi-tenant hotel food ordering system.

## 🚀 Features

- **Multi-Tenancy**: Support for multiple hotels with isolated data
- **Authentication**: JWT-based auth for Hotel Admins and Super Admins
- **Subscription Management**: Automated subscription validation
- **Image Uploads**: Multer-based image handling for menu items and hotel branding
- **RESTful API**: Clean, well-documented API endpoints
- **Security**: Bcrypt password hashing, JWT tokens, role-based access control

## 📋 Prerequisites

- Node.js (v14 or higher)
- MySQL (v5.7 or higher)
- npm or yarn

## 🛠️ Installation

### 1. Install Dependencies

```bash
npm install
```

### 2. Configure Environment

Copy `.env.example` to `.env` and update the values:

```bash
cp .env.example .env
```

Edit `.env` with your MySQL credentials:

```env
DB_HOST=localhost
DB_PORT=3306
DB_NAME=hotel_ordering_db
DB_USER=root
DB_PASSWORD=your_mysql_password
JWT_SECRET=your_secret_key_here
```

### 3. Create Database

Create a MySQL database:

```sql
CREATE DATABASE hotel_ordering_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

### 4. Initialize Database

Run the initialization script to create tables and seed data:

```bash
npm run init-db
```

This will create:
- All required tables
- Super Admin account (superadmin@hotel.com / SuperAdmin@123)
- Demo Hotel account (demo@hotel.com / demo123)

## 🎯 Running the Server

### Development Mode (with auto-restart)

```bash
npm run dev
```

### Production Mode

```bash
npm start
```

The server will start on `http://localhost:5000`

## 📚 API Documentation

### Public Endpoints (No Auth Required)

#### Get Hotel Info
```
GET /api/public/:hotelId/info
```

#### Get Menu Items
```
GET /api/public/:hotelId/menu
```

#### Get Categories
```
GET /api/public/:hotelId/categories
```

#### Get Active Offers
```
GET /api/public/:hotelId/offers
```

### Authentication Endpoints

#### Hotel Admin Login
```
POST /api/auth/login
Body: { "email": "demo@hotel.com", "password": "demo123" }
```

#### Super Admin Login
```
POST /api/auth/super-login
Body: { "email": "superadmin@hotel.com", "password": "SuperAdmin@123" }
```

### Hotel Admin Endpoints (Requires JWT Token)

Add header: `Authorization: Bearer <token>`

#### Dashboard Stats
```
GET /api/admin/dashboard
```

#### Menu Management
```
GET /api/admin/menu
POST /api/admin/menu (multipart/form-data with image)
PUT /api/admin/menu/:id
DELETE /api/admin/menu/:id
```

#### Category Management
```
GET /api/admin/categories
POST /api/admin/categories
PUT /api/admin/categories/:id
DELETE /api/admin/categories/:id
```

#### Offer Management
```
GET /api/admin/offers
POST /api/admin/offers
PUT /api/admin/offers/:id
DELETE /api/admin/offers/:id
```

#### Hotel Settings
```
PUT /api/admin/settings (multipart/form-data with logo/banner)
```

### Super Admin Endpoints (Requires Super Admin Token)

#### Hotel Management
```
GET /api/super/hotels
POST /api/super/hotels
PUT /api/super/hotels/:id/subscription
PUT /api/super/hotels/:id/status
DELETE /api/super/hotels/:id
```

## 🗄️ Database Schema

### Tables

- **super_admins**: System administrators
- **hotels**: Hotel accounts with subscription info
- **categories**: Menu categories (scoped by hotel)
- **menu_items**: Food items (scoped by hotel)
- **offers**: Promotional offers (scoped by hotel)

## 🔐 Security Features

- Password hashing with bcrypt (10 rounds)
- JWT token authentication
- Role-based access control (Hotel Admin / Super Admin)
- Subscription validation middleware
- Multi-tenancy isolation (hotels can only access their own data)
- File upload validation (images only, 5MB limit)

## 📁 Project Structure

```
Backend/
├── config/
│   └── database.js          # Sequelize configuration
├── controllers/
│   ├── authController.js    # Login handlers
│   ├── publicController.js  # Public endpoints
│   ├── menuController.js    # Menu public endpoints
│   ├── adminController.js   # Hotel admin CRUD
│   ├── hotelController.js   # Hotel settings
│   └── superAdminController.js # Super admin operations
├── middleware/
│   ├── auth.js              # JWT authentication
│   └── upload.js            # Multer file upload
├── models/
│   ├── SuperAdmin.js
│   ├── Hotel.js
│   ├── Category.js
│   ├── MenuItem.js
│   ├── Offer.js
│   └── index.js
├── routes/
│   ├── authRoutes.js
│   ├── publicRoutes.js
│   ├── adminRoutes.js
│   └── superAdminRoutes.js
├── scripts/
│   └── initDatabase.js      # Database initialization
├── public/
│   └── uploads/             # Uploaded images
├── .env                     # Environment variables
├── server.js                # Main application entry
└── package.json
```

## 🧪 Testing

### Test Super Admin Login

```bash
curl -X POST http://localhost:5000/api/auth/super-login \
  -H "Content-Type: application/json" \
  -d '{"email":"superadmin@hotel.com","password":"SuperAdmin@123"}'
```

### Test Hotel Admin Login

```bash
curl -X POST http://localhost:5000/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{"email":"demo@hotel.com","password":"demo123"}'
```

## 🚢 Deployment

### Environment Variables for Production

- Set `NODE_ENV=production`
- Use strong `JWT_SECRET`
- Update `FRONTEND_URL` to your production frontend URL
- Use environment-specific database credentials

### Recommended Platforms

- **Heroku**: Easy deployment with MySQL add-on
- **DigitalOcean**: App Platform or Droplet
- **AWS**: EC2 + RDS
- **Render**: Free tier available

## 📝 Notes

- Images are stored locally in `public/uploads/`. For production, consider using AWS S3 or similar cloud storage.
- Database uses `sync()` for development. Use migrations (Sequelize CLI) for production.
- CORS is configured for `http://localhost:5173` by default. Update for production.

## 🤝 Support

For issues or questions, please create an issue in the repository.

## 📄 License

MIT
